/**
 * \file gabidulin.c
 * \brief Implementation of gabidulin.h
 *
 * The decoding algorithm provided is based on q_polynomials reconstruction, see \cite gabidulin:welch and \cite gabidulin:generalized for details.
 *
 */

#include "parameters.h"
#include "gabidulin.h"
#include "q_polynomial.h"
#include "ffi_field.h"
#include "ffi_elt.h"
#include "ffi_vec.h"


/** 
 * \fn gabidulin_code* gabidulin_code_init(ffi_field field, const ffi_vec g, unsigned int k, unsigned int n)
 * \brief This function initializes a gabidulin code
 *
 * \param[in] field Finite field over which the code is defined
 * \param[in] g Generator vector defining the code
 * \param[in] k Size of vectors representing messages
 * \param[in] n Size of vetors representing codewords
 * \return Gabidulin code
 */
gabidulin_code* gabidulin_code_init(ffi_field field, const ffi_vec g, unsigned int k, unsigned int n) {
  gabidulin_code* gc = (gabidulin_code*) malloc(sizeof(gabidulin_code));

  *(gc->field) = *field;
  ffi_vec_init(field, &(gc->g), n); 
  ffi_vec_set(field, gc->g, g, n);
  gc->k = k;
  gc->n = n;

  return gc;
}



/** 
 * \fn void gabidulin_code_clear(gabidulin_code* gc)
 * \brief This function clears the allocated memory for a Gabidulin code 
 *
 * \param[in] gc Gabidulin code
 */
void gabidulin_code_clear(gabidulin_code* gc) {
  ffi_vec_clear(gc->field, gc->g, gc->n);
  free(gc);
}



/** 
 * \fn void gabidulin_code_encode(ffi_vec c, gabidulin_code* gc, const ffi_vec m)
 * \brief This function encodes a message into a codeword
 *
 * \param[out] c Vector of size <b>n</b> representing a codeword
 * \param[in] gc Gabidulin code
 * \param[in] m Vector of size <b>k</b> representing a message
 */
void gabidulin_code_encode(ffi_vec c, gabidulin_code* gc, const ffi_vec m) {
  // Compute generator matrix
  ffi_elt matrix[gc->k][gc->n];

  int q = ffi_field_get_characteristic(gc->field);
  for(unsigned int j = 0 ; j < gc->n ; ++j) {
    ffi_elt_init(gc->field, matrix[0][j]);
    ffi_elt_set(gc->field, matrix[0][j], gc->g[j]);
    for(unsigned int i = 1 ; i < gc->k ; ++i) {
      ffi_elt_init(gc->field, matrix[i][j]);
      ffi_elt_pow(gc->field, matrix[i][j], matrix[i-1][j], q);
    }
  }

  // Transform generator matrix in systematic form
  ffi_elt inv, tmp1, tmp2;
  ffi_elt_init(gc->field, inv);
  ffi_elt_init(gc->field, tmp1);
  ffi_elt_init(gc->field, tmp2);

  for(unsigned int i = 0 ; i < gc->k ; ++i) {
    ffi_elt_inv(gc->field, inv, matrix[i][i]);
    for(unsigned int j = i ; j < gc->n ; ++j) {
      ffi_elt_mul(gc->field, matrix[i][j], matrix[i][j], inv);
    }

    for(unsigned int k = 0 ; k < gc->k ; ++k) {
      if(k != i) {
        ffi_elt_set(gc->field, tmp1, matrix[k][i]);
        for(unsigned int j = i ; j < gc->n ; ++j) {
          ffi_elt_mul(gc->field, tmp2, matrix[i][j], tmp1);
          // ffi_elt_neg(gc->field, tmp2, tmp2);
          ffi_elt_add(gc->field, matrix[k][j], matrix[k][j], tmp2);
        }
      }
    }
  }
    
  // Encode message
  ffi_elt_set_zero(gc->field, tmp1);
  ffi_vec_set_zero(gc->field, c, gc->n);
  for(unsigned int i = 0 ; i < gc->k ; ++i) {
    for(unsigned int j = 0 ; j < gc->n ; ++j) {
      ffi_elt_mul(gc->field, tmp1, m[i], matrix[i][j]);
      ffi_elt_add(gc->field, c[j], c[j], tmp1);
    }
  }

  #ifdef VERBOSE
    printf("\n\n# Gabidulin Encoding - Begin #");
    printf("\n\ng: "); ffi_vec_print(gc->field, gc->g, PARAM_N);

    printf("\n\nmatrix: ");
    for(unsigned int i = 0 ; i < gc->k ; ++i) {
      for(unsigned int j = 0 ; j < gc->n ; ++j) {
        printf(" ");
        ffi_elt_print(gc->field, matrix[i][j]);
      }
      printf("\n");
    }

    printf("\ncodeword: "); ffi_vec_print(gc->field, c, PARAM_N);
    printf("\n\n# Gabidulin Encoding - End #");
  #endif
}



/** 
 * \fn void gabidulin_code_decode(ffi_vec m, gabidulin_code* gc, const ffi_vec y)
 * \brief This function decodes a word
 *
 * As explained in the supporting documentation, the provided decoding algorithm works as follows (see \cite gabidulin:welch and \cite gabidulin:generalized for details):
 *   1. Find a solution (<b>V</b>, <b>N</b>) of the q-polynomial Reconstruction2(<b>y</b>, <b>gc->g</b>, gc->k, (gc->n - gc->k)/2) problem using \cite gabidulin:generalized (section 4, algorithm 5) ;
 *   2. Find <b>f</b> by computing the left euclidean division of <b>N</b> by <b>V</b> ;
 *   3. Retrieve the codeword <b>c</b> by evaluating <b>f</b> in <b>gc->g</b> and get the message <b>m</b> as the k first coordinates of <b>c</b>.
 *
 * \param[out] m Vector of size <b>k</b> representing a message
 * \param[in] gc Gabidulin code
 * \param[in] y Vector of size <b>n</b> representing a word to decode
 */
void gabidulin_code_decode(ffi_vec m, gabidulin_code* gc, const ffi_vec y) {

  /*  
   *  Step 1: Solving the q-polynomial reconstruction2 problem 
   */
  q_polynomial* N0 = q_polynomial_init(gc->field, gc->n - 1);
  q_polynomial* N1 = q_polynomial_init(gc->field, gc->n - 1);
  q_polynomial* V0 = q_polynomial_init(gc->field, gc->n - 1);
  q_polynomial* V1 = q_polynomial_init(gc->field, gc->n - 1);

  q_polynomial* qtmp1 = q_polynomial_init(gc->field, gc->n - 1);
  q_polynomial* qtmp2 = q_polynomial_init(gc->field, gc->n - 1);
  q_polynomial* qtmp3 = q_polynomial_init(gc->field, gc->n - 1);
  q_polynomial* qtmp4 = q_polynomial_init(gc->field, gc->n - 1);

  ffi_vec g, u0, u1, u_tmp;
  ffi_vec_init(gc->field, &g, gc->n);
  ffi_vec_init(gc->field, &u0, gc->n);
  ffi_vec_init(gc->field, &u1, gc->n);
  ffi_vec_init(gc->field, &u_tmp, gc->n);
  ffi_vec_set(gc->field, g, gc->g, gc->n);

  ffi_elt e1, e2, tmp1, tmp2;
  ffi_elt_init(gc->field, e1);
  ffi_elt_init(gc->field, e2);
  ffi_elt_init(gc->field, tmp1);
  ffi_elt_init(gc->field, tmp2);

  int exponent = ffi_field_get_characteristic(gc->field);


  // Initialization step
  
  // N0(g[i]) = 0
  // N1(g[i]) = y[i]
  
  q_polynomial_set_interpolate_zero(N0, g, gc->k);
  q_polynomial_set_interpolate_vect(N1, g, y, gc->k);
  q_polynomial_set_zero(V0);
  q_polynomial_set_one(V1);

  q_polynomial_set_zero(qtmp1);
  q_polynomial_set_zero(qtmp2);
  q_polynomial_set_zero(qtmp3);
  q_polynomial_set_zero(qtmp4);

  for(unsigned int i = 0 ; i < gc->n ; ++i) {

    // u0[i] = N0(g[i]) - V0(y[i])
    // u1[i] = N1(g[i]) - V1(y[i])
  
    q_polynomial_evaluate(tmp1, N0, g[i]);
    q_polynomial_evaluate(u0[i], V0, y[i]);
    // ffi_elt_neg(gc->field, u0[i], u0[i]);
    ffi_elt_add(gc->field, u0[i], tmp1, u0[i]);

    q_polynomial_evaluate(tmp1, N1, g[i]);
    q_polynomial_evaluate(u1[i], V1, y[i]);
    // ffi_elt_neg(gc->field, u1[i], u1[i]);
    ffi_elt_add(gc->field, u1[i], tmp1, u1[i]);
  }

  #ifdef VERBOSE
    printf("\n\n# Gabidulin Decoding - Begin #");
    printf("\n\ng: "); ffi_vec_print(gc->field, gc->g, PARAM_N);
    printf("\n\nN0 (init): "); q_polynomial_print(N0);
    printf("\nN1 (init): "); q_polynomial_print(N1);
    printf("\nV0 (init): "); q_polynomial_print(V0);
    printf("\nV1 (init): "); q_polynomial_print(V1);
  #endif


  // Interpolation step 
  int updateType = -1;
  for(unsigned int i = gc->k ; i < gc->n ; ++i) {

    unsigned int j = i;
    while(ffi_elt_is_zero(gc->field, u0[j]) == 0 && 
          ffi_elt_is_zero(gc->field, u1[j]) == 1 && 
          j < gc->n) j++;
    
    if(j == gc->n) {
      break;
    } else {
      if(i != j) {
        // Permutation of the coordinates of positions i and j
        ffi_elt_set(gc->field, tmp1, y[i]);
        ffi_elt_set(gc->field, y[i], y[j]);
        ffi_elt_set(gc->field, y[j], tmp1);

        ffi_elt_set(gc->field, tmp1, g[i]);
        ffi_elt_set(gc->field, g[i], g[j]);
        ffi_elt_set(gc->field, g[j], tmp1);

        ffi_elt_set(gc->field, tmp1, u0[i]);
        ffi_elt_set(gc->field, u0[i], u0[j]);
        ffi_elt_set(gc->field, u0[j], tmp1);

        ffi_elt_set(gc->field, tmp1, u1[i]);
        ffi_elt_set(gc->field, u1[i], u1[j]);
        ffi_elt_set(gc->field, u1[j], tmp1);
      }
    }

    // Update q_polynomials according to discrepancies
    if(ffi_elt_is_zero(gc->field, u1[i]) != 1) {
      updateType = 1;

      // e1 = - u1[i]^q / u1[i] 
      // e2 = - u0[i] / u1[i]
      // N0' = N1^q - e1.N1
      // V0' = V1^q - e1.V1
      // N1' = N0 - e2.N1 
      // V1' = V0 - e2.V1
      
      ffi_elt_inv(gc->field, tmp1, u1[i]);

      ffi_elt_pow(gc->field, e1, u1[i], exponent);
      ffi_elt_mul(gc->field, e1, tmp1, e1);
      // ffi_elt_neg(gc->field, e1, e1);

      ffi_elt_mul(gc->field, e2, u0[i], tmp1);
      // ffi_elt_neg(gc->field, e2, e2);


      q_polynomial_scalar_mul(qtmp1, N1, e1);
      // q_polynomial_neg(qtmp1, qtmp1);
      q_polynomial_qexp(qtmp2, N1);
      q_polynomial_scalar_mul(qtmp3, V1, e1);
      // q_polynomial_neg(qtmp3, qtmp3);
      q_polynomial_qexp(qtmp4, V1);

      q_polynomial_scalar_mul(N1, N1, e2);
      q_polynomial_add(N1, N0, N1);

      q_polynomial_scalar_mul(V1, V1, e2);
      q_polynomial_add(V1, V0, V1);
      
      q_polynomial_add(N0, qtmp1, qtmp2);
      q_polynomial_add(V0, qtmp3, qtmp4);
    } 

    if(ffi_elt_is_zero(gc->field, u0[i]) == 1 && ffi_elt_is_zero(gc->field, u1[i]) == 1) {
      updateType = 2;

      // N0' = N1^q 
      // V0' = V1^q
      // N1' = N0 
      // V1' = V0 
      
      q_polynomial_qexp(qtmp1, N1);
      q_polynomial_qexp(qtmp2, V1);

      q_polynomial_set(N1, N0);
      q_polynomial_set(V1, V0);
      q_polynomial_set(N0, qtmp1);
      q_polynomial_set(V0, qtmp2);
    } 


    // Update discrepancies
    for(unsigned int k = i + 1 ; k < gc->n ; ++k) {
      if(updateType == 1) {

        // u0[k]' = u1[k]^q - e1.u1[k]
        // u1[k]' = u0[k] - e2.u1[k] 
      
        ffi_elt_mul(gc->field, tmp1, e1, u1[k]);
        // ffi_elt_neg(gc->field, tmp1, tmp1);
        ffi_elt_pow(gc->field, tmp2, u1[k], exponent);

        ffi_elt_mul(gc->field, u1[k], e2, u1[k]);
        // ffi_elt_neg(gc->field, u1[k], u1[k]);
        ffi_elt_add(gc->field, u1[k], u0[k], u1[k]);

        ffi_elt_add(gc->field, u0[k], tmp1, tmp2);
      } 
      
      if(updateType == 2) {

        // u0[k]' = u0[k]
        // u1[k]' = u1[k]^q
        
        ffi_elt_pow(gc->field, u1[k], u1[k], exponent);
      }
    }

    #ifdef VERBOSE
      printf("\nN0 (%i): ", i); q_polynomial_print(N0);
      printf("\nN1 (%i): ", i); q_polynomial_print(N1);
      printf("\nV0 (%i): ", i); q_polynomial_print(V0);
      printf("\nV1 (%i): ", i); q_polynomial_print(V1);
    #endif
  }



  /*  
   *  Step 2: Computing f (qtmp1) as the left euclidean division of N by V
   */
  q_polynomial_left_div(qtmp1, qtmp2, N1, V1);



  /*  
   *  Step 3: Decoding the message by evaluating f (qtmp1) in g
   */
  for(unsigned int i = 0 ; i < gc->k ; ++i) {
    q_polynomial_evaluate(m[i], qtmp1, g[i]);
  }


  #ifdef VERBOSE
    printf("\nquotient: "); q_polynomial_print(qtmp1);
    printf("\nremainder: "); q_polynomial_print(qtmp2);
    printf("\nmu: "); ffi_vec_print(gc->field, m, PARAM_K);
    printf("\n\n# Gabidulin Decoding - End #");
  #endif


  ffi_elt_clear(gc->field, e1);
  ffi_elt_clear(gc->field, e2);
  ffi_elt_clear(gc->field, tmp1);
  ffi_elt_clear(gc->field, tmp2);

  ffi_vec_clear(gc->field, g, gc->n);
  ffi_vec_clear(gc->field, u0, gc->n);
  ffi_vec_clear(gc->field, u1, gc->n);
  ffi_vec_clear(gc->field, u_tmp, gc->n);

  q_polynomial_clear(N0);
  q_polynomial_clear(N1);
  q_polynomial_clear(V0);
  q_polynomial_clear(V1);
  q_polynomial_clear(qtmp1);
  q_polynomial_clear(qtmp2);
  q_polynomial_clear(qtmp3);
  q_polynomial_clear(qtmp4);
}

